<?php

namespace Domain\Hub4all\API\Service;

use App\Application\Integrator\ApiClient;
use Carbon\Carbon;
use Domain\Integration\Models\IntegrationModel;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Log;

class Hub4allHttpClient
{
    private const CODE = "hub4all";
    private const HOST_DEV = "https://app.hub4all.com.br/api";
    private const HOST_LIVE = "https://app.hub4all.com.br/api";
    public array $credentials = [];

    public function __construct()
    {
        self::validateAccessToken();
        $this->credentials = self::getCredentials();

    }

    #region Authentication
    private function authentication()
    {
        $data = [
            'email' => env('HUB4ALL_USER'),
            'password' => env('HUB4ALL_PASSWORD')
        ];

        $response = $this->dispatch("post", "/auth/login", $data);

        IntegrationModel::query()
            ->where('code', '=', self::CODE)
            ->update([
                'credentials' => serialize($response->json())
            ]);
    }

    private function validateAccessToken()
    {
        $integration = IntegrationModel::query()->firstOrCreate(
            [
                'code' => self::CODE
            ],
            [
                'name' => 'Hub4all',
            ]
        );


        $today = new \DateTime();
        $dateDiff = $today->diff(new \DateTime($integration->updated_at));

        if (empty($integration->credentials) || $integration->updated_at->lessThan(Carbon::now()->subHours(6))){
            $this->authentication();
        }
    }

    private function getCredentials(): array
    {
        $integration = IntegrationModel::query()
            ->where('code', '=', self::CODE)
            ->first();

        return unserialize($integration->credentials);
    }
    #endregion

    #region Orders
    public function ordersGetAll(array $filters = [])
    {
        $queryParams = http_build_query($filters);

        $response = $this->dispatch("get", "/v2/orders?".$queryParams, []);

        return $response->json();
    }

    public function orderProcess(int $orderId)
    {
        $response = $this->dispatch("post", "/v2/orders/process/".$orderId);

        return $response->json();
    }

    public function orderInvoice(int $hub4allOrderId, array $data)
    {
        $response = $this->dispatch("post", "/v2/orders/invoice/".$hub4allOrderId, $data);

        return $response->json();
    }

    public function orderCancel(int $hub4allOrderId)
    {
        $response = $this->dispatch("post", "/v2/orders/cancel/".$hub4allOrderId);

        return $response->json();
    }

    public function orderFinish(int $hub4allOrderId)
    {
        $response = $this->dispatch("post", "/v2/orders/finish/".$hub4allOrderId);

        return $response->json();
    }
    #endregion

    #region Helpers
    /**
     * @param string $method
     * @param string $endpoint
     * @param array $data
     * @return Response
     * @throws \Exception
     */
    public function dispatch(string $method, string $endpoint, array $data = []): Response
    {
        $host = self::HOST_DEV;

        if(env("APP_ENV") == 'production'){
            $host = self::HOST_LIVE;
        }

        $headers = [];
        if($endpoint != '/auth/login'){
            $headers = [
                'Authorization' => "Bearer ". $this->credentials['access_token'],
                'Content-Type' => "application/json"
            ];
        }

        $url = $this->mountUrlApi($host, $endpoint);

        $response = (new ApiClient(
            $headers,
            $method,
            $url,
            $data
        ))->dispatch();

        Log::channel('hub4all')->info('Request para hub4all.', [
            'url' => $url,
            'response' => $response->json(),
            'endpoint' => $endpoint,
            'data' => $data,
            'method' => $method
        ]);

        if(!$response->successful() || !$response->json()['success']){

//            Log::channel('hub4all')->info('Erro ao obter atualização de pedido.', [
//                'url' => $url,
//                'response' => $response->json(),
//                'endpoint' => $endpoint,
//                'data' => $data,
//                'method' => $method
//            ]);

            throw new \Exception($response->body());
        }

        return $response;
    }

    private function mountUrlApi(string $host, string $endpoint): string
    {
        if (!str_ends_with($host, '/')) {
            $host .= '/';
        }

        if (str_starts_with($endpoint, '/')) {
            $endpoint = substr($endpoint, 1);
        }

        return "{$host}{$endpoint}";
    }
    #endregion
}
