<?php

namespace Domain\Order\Service;

use Domain\Hub4all\API\Service\Hub4allHttpClient;
use Domain\Order\Models\OrderModel;
use Domain\Protheus\API\Services\ProtheusHttpClient;
use Domain\Protheus\API\Services\ProtheusService;
use Illuminate\Support\Facades\Log;

class OrderService
{
    private ProtheusHttpClient $protheusHttpClient;
    private Hub4allHttpClient $hub4allHttpClient;

    public function __construct()
    {
        $this->protheusHttpClient = new ProtheusHttpClient();
        $this->hub4allHttpClient = new Hub4allHttpClient();
    }

    public function importFromHub4all(array $orderItem)
    {
        try{
            $order = OrderModel::query()->where('hub_order_id', '=', $orderItem['id'])->first();

            if(!empty($order) && !empty($order->processed_at)){
                return;
            }

            OrderModel::query()->updateOrCreate([
                'hub_order_id' => $orderItem['id'],
            ], [
                'hub_order_id' => $orderItem['id'],
                'hub_payload' => serialize($orderItem),
                'processed_at' => !empty($orderItem['processedAt']) ? $orderItem['processedAt']['date'] : null,
                'billed_at' => !empty($orderItem['billedAt']) ? $orderItem['billedAt']['date'] : null,
                'sent_at' => !empty($orderItem['sendingAt']) ? $orderItem['sendingAt']['date'] : null,
                'finished_at' => !empty($orderItem['finishedAt']) ? $orderItem['finishedAt']['date'] : null,
                'canceled_at' => !empty($orderItem['canceledAt']) ? $orderItem['canceledAt']['date'] : null,
                'error_import' => null
            ]);

        }catch (\Exception $e){
            OrderModel::query()
                ->where('hub_order_id', '=', $orderItem['id'])
                ->update([
                    'error_import' => "Arquivo:{$e->getFile()} | Linha: {$e->getLine()} | Error: {$e->getMessage()}"
                ]);
        }
    }

    public function exportToProtheus(int $orderId)
    {
        $order = OrderModel::query()->findOrFail($orderId);

        try {
            $orderItem = unserialize($order->hub_payload);

            if(!empty($order->processed_at)){
                return;
            }

            $protheusService = new ProtheusService();
            $protheusOrderCreated = $protheusService->createOrder($orderItem);

            $this->hub4allHttpClient->orderProcess($orderItem['id']);

            $protheusOrderId = intval($protheusOrderCreated['message']);

            if($protheusOrderId === 0){
                throw new \Exception("Código do pedido criado no protheus não retornado");
            }

            OrderModel::query()
                ->where('hub_order_id', '=', $orderItem['id'])
                ->update([
                    'protheus_order_id' => $protheusOrderId,
                    'processed_at' => new \DateTime('now'),
                    'error_export' => null
                ]);
        }catch (\Throwable $e){
            OrderModel::query()
                ->where('id', $orderId)
                ->update([
                    'error_export' => "Error: {$e->getMessage()} | Arquivo:{$e->getFile()} | Linha: {$e->getLine()}"
                ]);
        }
    }

    public function checkOrderUpdates(string $protheusOrderId)
    {
        try {
            $orderGatewayModel = OrderModel::query()->where('protheus_order_id',$protheusOrderId)->first();
            $orderProtheusStatus = $this->protheusHttpClient->orderStatus($protheusOrderId);

            $code = $orderProtheusStatus['code'] ?? null;
            $nf = $orderProtheusStatus['nf'] ?? null;

            switch ($code) {
                case 'H':
                    if (!empty($nf) && empty($orderGatewayModel->billed_at)) {
                        $this->hub4allHttpClient->orderInvoice($orderGatewayModel->hub_order_id, [
                            'number' => $nf,
                            'issue_date' => (new \DateTime('now'))->format('Y-m-d'),
                        ]);

                        $orderGatewayModel->update([
                            'billed_at' => new \DateTime('now'),
                            'error_check_update' => null
                        ]);

                        Log::channel('order')->info("Pedido { $protheusOrderId } faturado");
                    } else {
                        Log::channel('order')->info("Pedido { $protheusOrderId } não contém dados da nota fiscal.");
                    }
                    break;

                case 'Z':
                    if (empty($orderGatewayModel->canceled_at)) {
                        $this->hub4allHttpClient->orderCancel($orderGatewayModel->hub_order_id);

                        $orderGatewayModel->update([
                            'canceled_at' => new \DateTime('now'),
                            'error_check_update' => null
                        ]);

                        Log::channel('order')->info("Pedido { $protheusOrderId } cancelado");
                    }

                    break;

                case 'J':
                    if (empty($orderGatewayModel->finished_at)) {
                        $this->hub4allHttpClient->orderFinish($orderGatewayModel->hub_order_id);

                        $orderGatewayModel->update([
                            'finished_at' => new \DateTime('now'),
                            'error_check_update' => null
                        ]);

                        Log::channel('order')->info("Pedido { $protheusOrderId } finalizado");
                    }
                    break;

                default:
                    Log::channel('order')->info('Pedido não precisa de ação', [
                        'protheus_order_id' => $protheusOrderId,
                        'order_status' => $orderProtheusStatus
                    ]);

                    break;
            }
        } catch (\Exception $e){
            OrderModel::query()
                ->where('protheus_order_id', '=', $protheusOrderId)
                ->update([
                    'error_check_update' => "Arquivo:{$e->getFile()} | Linha: {$e->getLine()} | Error: {$e->getMessage()}"
                ]);
        }

    }

}
